<?php
/*--------------------------------------------------------------
   GambioConversionRateTrend.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Core\Application\ValueObjects\UserPreferences;

class GambioConversionRateTrend extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.conversionrate.trend';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Konversionsrate (Trend)',
        self::LANGUAGE_CODE_ENGLISH => 'Conversion rate (Trend)'
    ];
    
    private const CATEGORY_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Konversion',
        self::LANGUAGE_CODE_ENGLISH => 'Conversion'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        UserPreferences $userPreferences,
        Connection $connection
    ) {
        $this->dataFactory     = $dataFactory;
        $this->userPreferences = $userPreferences;
        $this->connection      = $connection;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createAreaChart(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdown($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $categories    = [];
        $values        = [];
        $categoryTitle = self::CATEGORY_TITLE[$this->userPreferences->languageId()
                                              === self::LANGUAGE_ID_GERMAN ? self::LANGUAGE_CODE_GERMAN : self::LANGUAGE_CODE_ENGLISH];
        $timespan      = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)
                                                                               ->value());
        $parameters    = [
            ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
            ':startDate'        => $timespan->startDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
            ':endDate'          => $timespan->endDate()->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
        ];
        
        $orders = $this->connection->createQueryBuilder()
            ->select('UNIX_TIMESTAMP(DATE(date_purchased)) AS date, COUNT(*) AS orders')
            ->from('orders')
            ->where('orders_status NOT IN (:excludedOrderIds)')
            ->andWhere('date_purchased BETWEEN :startDate AND :endDate')
            ->groupBy('date')
            ->orderBy('date')
            ->setParameters($parameters)
            ->execute()
            ->fetchAll();
        
        $visitors = $this->connection->createQueryBuilder()
            ->select('UNIX_TIMESTAMP(DATE(gm_counter_date)) AS date, gm_counter_visits_total AS visitors')
            ->from('gm_counter_visits')
            ->where('gm_counter_date BETWEEN :startDate AND :endDate')
            ->orderBy('date')
            ->setParameters($parameters)
            ->execute()
            ->fetchAll();
        
        foreach ($visitors as $visitor) {
            $visitorDate  = $visitor['date'];
            $visitorCount = (float)$visitor['visitors'];
            
            $hasOrders = false;
            foreach ($orders as $order) {
                $orderDate  = $order['date'];
                $orderCount = (float)$order['orders'];
                
                if ($visitorDate !== $orderDate) {
                    continue;
                }
                
                $hasOrders = true;
                
                $categories[] = $this->dataFactory->useSerialData()->createCategory($orderDate);
                $values[]     = $this->dataFactory->useSerialData()->createItemValue(round(($orderCount / $visitorCount)
                                                                                           * 100));
            }
            
            if (!$hasOrders) {
                $categories[] = $this->dataFactory->useSerialData()->createCategory($visitorDate);
                $values[]     = $this->dataFactory->useSerialData()->createItemValue(0);
            }
        }
        
        return $this->dataFactory->useSerialData()->createSerialData($this->dataFactory->useSerialData()
                                                                         ->createCategories(...$categories),
                                                                     $this->dataFactory->useSerialData()
                                                                         ->createItems($this->dataFactory->useSerialData()
                                                                                           ->createItem($this->dataFactory->useSerialData()
                                                                                                            ->createItemTitle($categoryTitle),
                                                                                                        $this->dataFactory->useSerialData()
                                                                                                            ->createItemValues(...
                                                                                                                $values))));
    }
}
